/** @file   game.h
 * @brief   Initializes the game and runs the menupart.
 * @version $Revision: 1.2 $
 * @author  Tomi Lamminsaari
 */

#ifndef H_WWW_GAME_H
#define H_WWW_GAME_H

#include <allegro.h>
#include "eng2d.h"

/** @namespace WeWantWar
 * @par
 * -
 * @brief   All the sources of We Want War is located inside this namespace.
 * @author  Tomi Lamminsaari
 */
namespace WeWantWar {

// Forward declaration
class MenuPage;


/** @class  Game
 * @brief   Initializes the game and manages the menupart.
 * @version $Revision: 1.2 $
 * @author  Tomi Lamminsaari
 
 * Game-class represents the game. It is responssible of initializing
 * the game structures and displaying the menupart.
 * 
 * The 'main()'-function creates an instance of this class which then
 * takes the respossibility to init and run the game.
 * 
 * The <code>int init()</code>-method calls the initialization methods
 * of static classes like, <code>GameAnims, GfxPool, AnimPlayer, etc. </code>
 * 
 * The <code>void run()</code>-method then creates the MenuPage-objects and
 * creates the instances of <code>War</code>-class.
 */
class Game
{
public:


  ///
  /// Constructors, destructor and operators
  /// ======================================
  

  /** Constructor.
   */
  Game();
  
  /** Destructor.
   */
  ~Game();
  

  ///
  /// Public methods
  /// ==============

  /** Inits the game. Loads graphics, sounds and settings and creates
   * animatiotables.
   * @return    0 if initialization was ok. -1 if it fails.
   */
  int init();
  
  /** Runs the game. */
  void run();
  
  /** Cleans up and releases the resources allocated in 'init()'-method. */
  void cleanup();



protected:

  /** Changes the menupages. This controls the menustructures. */
  void doPageChange();
  
  /** Notification-menupage is different from the other pages. This manages
   * the page changes where Notification-window is involved. */
  void handleNotification();
  
  /** Creates new game starting from gamelevel 1*/
  void startNewGame();
  
  /** Deletes the current game. */
  void endCurrentGame();
  
  /** Continues the current game from next level. */
  void continueCurrentGame();
  
  /** Saves the current game.
   * @param     slot              To which saveslot we save the game.
   */
  void saveGame( int slot );
  
  /** Loads the game.
   * @param     slot              Index of the slot where we load the game.
   */
  void loadGame( int slot );
  
  /** Obsoled! */
  void playDemo();
  
  /** Plays the level.
   * @param     levelnum          Which level. 0 = the very first gamelevel.
   * @return    The reason why gameplay ended. See War-class to see the
   *            possible reasons.
   */
  int playLevel(int levelnum);

  /** Checks if player managed to get to the highscore table.
   * @return    <code>true</code> if player should write his name.
   */
  void checkForHighscores();
  
  /** Callback function for switch in case.
   */
  static void switchInCallback();
  
  /** Callback function for swicth out
   */
  static void switchOutCallback();

  ///
  /// Members
  /// =======

  /** Indicates when the mainloop should quit the game and return to windows. */
  bool  m_quitNow;
  /** Pointer to song we're currently playing. */
  eng2d::MP3File* m_titlesong;
  /** Pointer to an instance of MenuPage we're currently showing. */
  MenuPage*       m_pPage;
  
  /** Number of the next level. */
  int             m_nextLevel;
  
  
  static bool gameHasFocus;
  static void AudioPlayerUpdater();
  
private:

  Game(const Game& rO);
  Game& operator = (const Game& rO);
};

};  // end of namespace

#endif

/**
 * Version history
 * ===============
 * $Log: game.h,v $
 * Revision 1.2  2006/08/20 21:11:53  lamminsa
 * Switch background/foreground callbacks added. Timer callback
 * for MP3 player added.
 *
 * Revision 1.1.1.1  2006/01/21 23:02:41  lamminsa
 * no message
 *
 * Revision 1.0  2005-11-06 01:16:57+02  lamminsa
 * Initial revision
 *
 */
 

